#include "stdafx.h"
#include "wxTranslationHelper.h"
#include <wx/dir.h>
#include <wx/config.h>
#include <wx/fileconf.h>
#include <wx/filename.h>

wxTranslationHelper::wxTranslationHelper(wxApp & app, 
										 const wxString & search_path,
										 bool use_native_config)
: m_SearchPath(search_path), m_usingApp(true), 
m_ConfigPath(wxEmptyString), m_Locale(NULL), m_UseNativeConfig(use_native_config), m_language(wxLANGUAGE_UNKNOWN)
{   
	wxApp & m_App = app;

	if (search_path.IsEmpty()) {
		m_SearchPath = wxPathOnly(m_App.argv[0]);
	}
	m_appName = m_App.GetAppName();
	m_appPath = wxPathOnly(m_App.argv[0]);
}

wxTranslationHelper::wxTranslationHelper(const wxString &app, 
										 const wxString &search_path,
										 bool use_native_config)
: m_appName(app), m_SearchPath(search_path), m_usingApp(false), 
m_ConfigPath(wxEmptyString), m_Locale(NULL), m_UseNativeConfig(use_native_config)
{   
	if (search_path.IsEmpty()) {
		m_SearchPath = ".";
	}
}

wxTranslationHelper::~wxTranslationHelper()
{
	Save();
	if (m_Locale) {
		wxDELETE(m_Locale);
	}
}

wxLocale * wxTranslationHelper::GetLocale()
{
	return (m_Locale);
}

const wxString & wxTranslationHelper::GetSearchPath()
{
	return (m_SearchPath);
}

void wxTranslationHelper::SetSearchPath(wxString & value)
{
	m_SearchPath = value;
	if (m_SearchPath.IsEmpty()) {
		if (m_usingApp) {
			m_SearchPath = m_appPath;
		} else {
			m_SearchPath = ".";
		}
	}
}

const wxString & wxTranslationHelper::GetConfigPath()
{
	return (m_ConfigPath);
}

void wxTranslationHelper::SetConfigPath(wxString & value)
{
	m_ConfigPath = value;
}

bool wxTranslationHelper::Load()
{
	wxConfigBase * config;
	if (m_UseNativeConfig) {
		config = new wxConfig(m_appName);
	} else {
		config = new wxFileConfig(m_appName, wxEmptyString, m_ConfigPath);
	}
	long language;
	config->SetPath(wxT("wxTranslation"));
	config->Read(wxT("wxTranslationLanguage"), &language, wxLANGUAGE_ENGLISH);

	delete config;
	if (language == wxLANGUAGE_UNKNOWN) {
		return (false);
	}
	wxArrayString names;
	wxArrayLong identifiers;
	GetInstalledLanguages(names, identifiers);
	for (size_t i = 0; i < identifiers.Count(); i++) {
		if (identifiers[i] == language) {
			m_translations = new wxTranslations();
			// Set the translations to the desire language.  This is not the locale!
			m_language = (wxLanguage)identifiers[i];
			m_translations->SetLanguage(m_language);
			wxFileTranslationsLoader::AddCatalogLookupPathPrefix(m_SearchPath);
			m_translations->AddCatalog(m_appName);
			wxTranslations::Set(m_translations); // wxTranslations will own the object and delete it for us.
#if 0
			if (m_Locale) wxDELETE(m_Locale);
			m_Locale = new wxLocale;
			// Set C locale to English!  This is the locale, not the translations.
			m_Locale->Init(wxLANGUAGE_ENGLISH_US, 0);
#endif
			return (true);
		}
	}   
	return (false);
}

void wxTranslationHelper::Save(bool bReset)
{
	wxConfigBase * config;
	if (m_UseNativeConfig) {
		config = new wxConfig(m_appName);
	} else {
		config = new wxFileConfig(m_appName, wxEmptyString, m_ConfigPath);
	}
	long language = wxLANGUAGE_UNKNOWN;
	if (!bReset) {
		if (m_language != wxLANGUAGE_UNKNOWN) {
			language = m_language;
		}
	}
	config->DeleteEntry(wxT("wxTranslation"));
	config->SetPath(wxT("wxTranslation"));
	config->Write(wxT("wxTranslationLanguage"), language);
	config->Flush();
	delete config;
}

void wxTranslationHelper::GetInstalledLanguages(wxArrayString & names, 
												wxArrayLong & identifiers)
{
	names.Clear();
	identifiers.Clear();    
	wxString filename;  
	const wxLanguageInfo * langinfo;    
	wxString name = wxLocale::GetLanguageName(wxLANGUAGE_DEFAULT);
	if (!name.IsEmpty()) {
		names.Add(_("Default"));
		identifiers.Add(wxLANGUAGE_DEFAULT);        
	}
	if (!wxDir::Exists(m_SearchPath)) {
		wxLogTrace(wxTraceMask(), wxT("Directory %s DOES NOT EXIST !!!"),
				   m_SearchPath.GetData());
		return;
	}
	wxDir dir(m_SearchPath);
#ifdef __WXMSW__
#define MASK wxT("*")
#else
#define MASK wxT("*")
#endif

	bool cont = false;
	for (cont = dir.GetFirst(&filename, MASK, wxDIR_DIRS); cont == true; cont = dir.GetNext(&filename)) {
		langinfo = wxLocale::FindLanguageInfo(filename);
		if (langinfo != NULL) {
			wxLogTrace(wxTraceMask(), wxT("SEARCHING FOR %s"),
					   wxString(dir.GetName()+wxFileName::GetPathSeparator()+
								filename+wxFileName::GetPathSeparator()+
								m_appName + wxT(".mo")).GetData());
			if (wxFileExists(dir.GetName()+wxFileName::GetPathSeparator()+
							 filename+wxFileName::GetPathSeparator()+
							 m_appName + wxT(".mo"))) {
				names.Add(langinfo->Description);
				identifiers.Add(langinfo->Language);
			}
		}
	}
}

bool wxTranslationHelper::AskUserForLanguage(wxArrayString & names, 
											 wxArrayLong & identifiers)
{
	wxCHECK_MSG(names.Count() == identifiers.Count(), false, 
				wxT("Array of language names and identifiers should have the same size."));
	long index = wxGetSingleChoiceIndex(_("Select the language"), 
										_("Language"), names);
	if (index != -1) {
		m_translations = new wxTranslations();
		// Set the translations to the desire language.  This is not the locale!
		m_language = (wxLanguage)identifiers[index];
		m_translations->SetLanguage(m_language);
		wxFileTranslationsLoader::AddCatalogLookupPathPrefix(m_SearchPath);
		wxLogTrace(wxTraceMask(), 
				   wxT("wxTranslationHelper: Path Prefix = \"%s\""), 
				   m_SearchPath.GetData());
		m_translations->AddCatalog(m_appName);
		wxLogTrace(wxTraceMask(), 
				   wxT("wxTranslationHelper: Catalog Name = \"%s\""), 
				   m_appName.GetData());
		wxTranslations::Set(m_translations); // wxTranslations will own the object and delete it for us.
#if 0
		if (m_Locale) wxDELETE(m_Locale);
		m_Locale = new wxLocale;
		// Set C locale to English!  This is the locale, not the translations.
		m_Locale->Init(wxLANGUAGE_ENGLISH_US, 0);
#endif
		return (true);
	}
	return (false);
}
